// GraphNav Popup Logic
document.addEventListener('DOMContentLoaded', () => {
    const loader = document.getElementById('loader');
    const refreshBtn = document.getElementById('refresh-btn');
    const configBtn = document.getElementById('config-btn');
    const layoutBtn = document.getElementById('layout-btn');

    if (configBtn && loader && refreshBtn && layoutBtn) {
        setupApp(loader, refreshBtn, configBtn, layoutBtn);
    } else {
        console.error("GraphNav: Essential UI elements not found.");
    }
});

async function setupApp(loader, refreshBtn, configBtn, layoutBtn) {
    // Theme & Layout state
    let lastData = null;
    const savedTheme = localStorage.getItem('theme') || 'light';
    const savedLayout = localStorage.getItem('layout') || 'force';

    // Theme initialization
    document.documentElement.setAttribute('data-theme', savedTheme);
    updateConfigIcon(configBtn, savedTheme);
    updateLayoutIcon(layoutBtn, savedLayout);

    // Event Listeners
    configBtn.addEventListener('click', () => {
        const currentTheme = document.documentElement.getAttribute('data-theme');
        const newTheme = currentTheme === 'light' ? 'dark' : 'light';
        document.documentElement.setAttribute('data-theme', newTheme);
        localStorage.setItem('theme', newTheme);
        updateConfigIcon(configBtn, newTheme);
        if (lastData) renderGraph(lastData);
    });

    layoutBtn.addEventListener('click', () => {
        const currentLayout = localStorage.getItem('layout') || 'force';
        const newLayout = currentLayout === 'force' ? 'tree' : 'force';
        localStorage.setItem('layout', newLayout);
        updateLayoutIcon(layoutBtn, newLayout);
        if (lastData) renderGraph(lastData);
    });

    refreshBtn.addEventListener('click', () => {
        loader.style.opacity = '1';
        loader.style.display = 'flex';
        init();
    });

    async function init() {
        try {
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            if (!tab.url || tab.url.startsWith('chrome://')) {
                showError(loader, "Cannot analyze this page.");
                return;
            }

            const results = await chrome.scripting.executeScript({
                target: { tabId: tab.id },
                func: () => {
                    const links = Array.from(document.querySelectorAll('a'))
                        .map(a => a.href)
                        .filter(href => href.startsWith('http'));
                    return {
                        links: [...new Set(links)],
                        origin: window.location.origin
                    };
                }
            });

            const { links, origin } = results[0].result;
            const sitemapLinks = await fetchSitemap(origin);
            const allLinks = [...new Set([...links, ...sitemapLinks])];
            const treeData = buildTree(allLinks, origin);
            lastData = treeData;
            renderGraph(treeData);

            loader.style.opacity = '0';
            setTimeout(() => loader.style.display = 'none', 300);
        } catch (err) {
            console.error("GraphNav Analysis Error:", err);
            showError(loader, `Analysis failed: ${err.message || "Unknown error"}`);
        }
    }

    // Start analysis
    init();
}

/** 
 * HELPER FUNCTIONS
 */

async function fetchSitemap(url) {
    try {
        const response = await fetch(url.endsWith('.xml') ? url : `${url}/sitemap.xml`);
        if (!response.ok) return [];
        const text = await response.text();
        const parser = new DOMParser();
        const xmlDoc = parser.parseFromString(text, "text/xml");
        const sitemaps = Array.from(xmlDoc.querySelectorAll('sitemap > loc')).map(el => el.textContent.trim());
        if (sitemaps.length > 0) {
            const nestedResults = await Promise.all(sitemaps.map(sm => fetchSitemap(sm)));
            return nestedResults.flat();
        }
        return Array.from(xmlDoc.querySelectorAll('url > loc')).map(el => el.textContent.trim());
    } catch (e) {
        return [];
    }
}

function buildTree(links, origin) {
    const root = { name: new URL(origin).hostname, children: [], url: origin };
    const nodeMap = { '': root };
    links.forEach(link => {
        try {
            const url = new URL(link);
            if (url.origin !== origin) return;
            const pathParts = url.pathname.split('/').filter(p => p);
            let currPath = '';
            let currNode = root;
            pathParts.forEach((part, index) => {
                const nextPath = currPath + '/' + part;
                if (!nodeMap[nextPath]) {
                    const newNode = { name: part, children: [], url: index === pathParts.length - 1 ? link : null };
                    currNode.children.push(newNode);
                    nodeMap[nextPath] = newNode;
                }
                currNode = nodeMap[nextPath];
                currPath = nextPath;
            });
            if (url.pathname !== '/' && !currNode.url) currNode.url = link;
        } catch (e) { }
    });
    return root;
}

function renderGraph(data) {
    const layout = localStorage.getItem('layout') || 'force';
    if (layout === 'tree') renderTreeGraph(data);
    else renderForceGraph(data);
}

function renderForceGraph(data) {
    const width = 600, height = 400;
    const svg = d3.select("#graph");
    svg.selectAll("*").remove();
    const container = svg.append("g");
    const root = d3.hierarchy(data);
    const nodes = root.descendants();
    const links = root.links();
    const simulation = d3.forceSimulation(nodes)
        .force("link", d3.forceLink(links).id(d => d.id).distance(50).strength(1))
        .force("charge", d3.forceManyBody().strength(-200))
        .force("center", d3.forceCenter(width / 2, height / 2))
        .force("collision", d3.forceCollide().radius(40));
    const link = container.append("g").selectAll("line").data(links).enter().append("line").attr("class", "link");
    const node = container.append("g").selectAll("g").data(nodes).enter().append("g")
        .attr("class", d => "node " + (d.data.url ? "clickable" : "parent"))
        .call(d3.drag().on("start", (e, d) => { if (!e.active) simulation.alphaTarget(0.3).restart(); d.fx = d.x; d.fy = d.y; })
            .on("drag", (e, d) => { d.fx = e.x; d.fy = e.y; })
            .on("end", (e, d) => { if (!e.active) simulation.alphaTarget(0); d.fx = null; d.fy = null; }))
        .on("click", (event, d) => { if (d.data.url) chrome.tabs.update({ url: d.data.url }); });
    setupNodeContent(node);
    simulation.on("tick", () => {
        link.attr("x1", d => d.source.x).attr("y1", d => d.source.y).attr("x2", d => d.target.x).attr("y2", d => d.target.y);
        node.attr("transform", d => `translate(${d.x},${d.y})`);
    });
    svg.call(d3.zoom().scaleExtent([0.1, 5]).on("zoom", (event) => container.attr("transform", event.transform)));
}

function renderTreeGraph(data) {
    const width = 600, height = 400;
    const svg = d3.select("#graph");
    svg.selectAll("*").remove();
    const container = svg.append("g");
    const tree = d3.tree().nodeSize([40, 150]);
    const root = d3.hierarchy(data);
    tree(root);
    container.append("g").selectAll("path").data(root.links()).enter().append("path")
        .attr("class", "link").attr("fill", "none").attr("d", d3.linkHorizontal().x(d => d.y).y(d => d.x));
    const node = container.append("g").selectAll("g").data(root.descendants()).enter().append("g")
        .attr("class", d => "node " + (d.data.url ? "clickable" : "parent"))
        .attr("transform", d => `translate(${d.y},${d.x})`)
        .on("click", (event, d) => { if (d.data.url) chrome.tabs.update({ url: d.data.url }); });
    setupNodeContent(node);
    const zoom = d3.zoom().scaleExtent([0.1, 5]).on("zoom", (event) => container.attr("transform", event.transform));
    svg.call(zoom);
    svg.call(zoom.transform, d3.zoomIdentity.translate(80, height / 2).scale(0.8));
}

function setupNodeContent(node) {
    node.append("circle")
        .attr("r", d => d.depth === 0 ? 8 : (d.children ? 6 : 4))
        .attr("fill", d => d.data.url ? (d.depth === 0 ? "#818cf8" : "#38bdf8") : "#e2e8f0")
        .style("stroke-opacity", d => d.data.url ? 1 : 0.4)
        .style("opacity", d => d.data.url ? 1 : 0.6);
    node.append("text").attr("dy", 15).attr("text-anchor", "middle").text(d => d.data.name).style("opacity", 1)
        .style("font-weight", d => d.data.url ? "600" : "400");
    node.on("mouseover", function (event, d) {
        if (d.data.url) {
            d3.select(this).select("circle").transition().duration(200).attr("r", 10).attr("fill", "#fff").style("stroke", "var(--accent-color)");
            d3.select(this).select("text").transition().duration(200).style("fill", "var(--accent-color)");
        }
    }).on("mouseout", function (event, d) {
        d3.select(this).select("circle").transition().duration(200).attr("r", d => d.depth === 0 ? 8 : (d.children ? 6 : 4))
            .attr("fill", d => d.data.url ? (d.depth === 0 ? "#818cf8" : "#38bdf8") : "#e2e8f0").style("stroke", "var(--bg-color)");
        d3.select(this).select("text").transition().duration(200).style("fill", "var(--text-secondary)");
    });
}

function updateConfigIcon(btn, theme) {
    btn.innerHTML = theme === 'light'
        ? `<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 3a6 6 0 0 0 9 9 9 9 0 1 1-9-9Z"></path></svg>`
        : `<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="4"/><path d="M12 2v2"/><path d="M12 20v2"/><path d="m4.93 4.93 1.41 1.41"/><path d="m17.66 17.66 1.41 1.41"/><path d="M2 12h2"/><path d="M20 12h2"/><path d="m6.34 17.66-1.41 1.41"/><path d="m19.07 4.93-1.41 1.41"/></svg>`;
}

function updateLayoutIcon(btn, layout) {
    btn.innerHTML = layout === 'force'
        ? `<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="3"/><circle cx="6" cy="6" r="3"/><circle cx="18" cy="18" r="3"/><line x1="8.12" y1="8.12" x2="15.88" y2="15.88"/><line x1="9.88" y1="14.12" x2="14.12" y2="9.88"/></svg>`
        : `<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M3 12h18M3 6h18M3 18h18"/></svg>`;
}

function showError(loader, msg) { loader.innerHTML = `<p style="color: #ef4444">${msg}</p>`; }
